/******************************************************************************/
/*                                                                            */
/*  some simple menues using curses                                         */
/*                                                                            */
/*  Copyright (C) 1999 Roland Maetche  DD5RM                                  */
/*                                                                            */
/* This program is free software; you can redistribute it and/or modify       */
/* it under the terms of the GNU General Public License as published by       */
/* the Free Software Foundation; either version 2 of the License, or          */
/* (at your option) any later version.                                        */
/*                                                                            */
/* Note: these functions are *not* optimized for minimum data transfer !      */
/*                                                                            */
/******************************************************************************/

/******************************************************************************/
/* includes                                                                   */
/******************************************************************************/
#include <curses.h>
#include <stdlib.h>
#include <sched.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <malloc.h>

#include "menu.h"

/******************************************************************************/
/* definitions                                                                */
/******************************************************************************/
#define KB_ESC			27
#define KB_ESC2			91
#define KB_ESC_UP		'A'
#define	KB_ESC_DOWN		'B'
#define	KB_ESC_RIGHT	'C'
#define	KB_ESC_LEFT		'D'

/******************************************************************************/
/* functions                                                                  */
/******************************************************************************/
/*** create a menu window ***/
Menu * create_menu( int rows, int cols, int top, int left, void (*updfunc)(void) )
{
	Menu *	me;
	
	me = malloc( sizeof( Menu ) );
	if( me != NULL ) {
		me->w = newwin( rows, cols, top, left );
		me->cur_mi = NULL;
		me->updfunc = updfunc;	
	}
	return me;
}

/*** refresh a menue (maybe update values ***/
void refresh_menu( Menu * m )
{
	if( m != NULL ) {
		if( m->updfunc != NULL ) m->updfunc();
		touchwin( m->w );
		wrefresh( m->w );
	}
} 

/*** add an item to menu ***/
void add_menuitem( Menu * me, int left, int top, char * text, char key, int (*func)(void * data), void * fdata )
{
	Menuitem *	mi;
	Menuitem *	cmi;
	char *		pd;

	mi = malloc( sizeof( Menuitem ) );
	mi->x = left;
	mi->y = top;
	mi->text = malloc( strlen( text ) + 3 );
	pd = mi->text;
	*pd++ = ' ';
	strcpy( pd, text );
	pd += strlen( text );
	*pd++ = ' ';
	*pd = '\0';
	mi->key = key;
	mi->func = func;
	mi->funcdata = fdata;
	mi->prev = mi;
	mi->next = mi;
	if( me->cur_mi == NULL ) me->cur_mi = mi;
	else {
		cmi = me->cur_mi;
		cmi->prev->next = mi;
		mi->prev = cmi->prev;
		cmi->prev = mi;
		mi->next = cmi;
	}
}

/*** delete menuitem ***/
void del_menuitem( Menuitem * mi )
{
	if( mi != mi->prev ) {
		mi->prev->next = mi->next;
		mi->next->prev = mi->prev;
	}
	free( mi->text );
	free( mi );
}

/*** delete menu ***/
void del_menu( Menu * me )
{
	Menuitem *	mi;
	
	while( me->cur_mi != NULL ) {
		mi = me->cur_mi->prev;
		del_menuitem( mi );
		if( mi == me->cur_mi ) me->cur_mi = NULL;
	}
	delwin( me->w );
	free( me );
}

/*** output text with underlined key ***/
void show_text( WINDOW * w, const char * text, char key )
{
	int	done;

	done = 0;
	while( *text != '\0' ) {
		if( ( *text == key ) && ( done == 0 ) ) {
			wattron( w, A_UNDERLINE );
			waddch( w, *text++ );
			wattroff( w, A_UNDERLINE );
			done = 1;
		}
		else waddch( w, *text++ );
	}
}

/*** show one menuitem ***/
void show_menuitem( WINDOW * w, Menuitem * mi, int attr )
{
	wmove( w, mi->y, mi->x );
	if( attr != 0 ) wattron( w, attr );
	show_text( w, mi->text, mi->key );
	if( attr != 0 ) wattroff( w, attr );
}

/*** show menue ***/
void show_menu( Menu * me )
{
	Menuitem *	mi;

	mi = me->cur_mi;
	do {
		show_menuitem( me->w, mi, ( ( mi == me->cur_mi ) ? A_BOLD : 0 ) );
		mi = mi->next;
	} while( mi != me->cur_mi );
	wmove( me->w, 0, 0 );
	wrefresh( me->w );
}

/*** find item in menu using specified key ***/
Menuitem * find_menuitem_key( Menuitem * start, char key )
{
	Menuitem *	mi;
	
	mi = start;
	do {
		if( toupper( mi->key ) == toupper( key ) ) return mi;
		mi = mi->next;
	} while( mi != start );
	return NULL;
}

/*** find item in menu using direction ***/
Menuitem * find_menuitem_dir( Menuitem * start, char dir )
{
	Menuitem *	mi;
	Menuitem *	found;
	int		diff;
	
	mi = start->next;
	found = NULL;
	diff = 999;
	while( mi != start ) {
		switch( dir ) {
		case KB_ESC_UP:
			if( mi->x == start->x ) {
				if( mi->y < start->y ) {
					if( ( start->y - mi->y ) < diff ) {
						diff = start->y -mi->y;
						found = mi;
					}
				}
			}
			break;
		case KB_ESC_DOWN:
			if( mi->x == start->x ) {
				if( mi->y > start->y ) {
					if( ( mi->y - start->y ) < diff ) {
						diff = mi->y - start->y;
						found = mi;
					}
				}
			}
			break;
		case KB_ESC_RIGHT:
			if( mi->y == start->y ) {
				if( mi->x > start->x ) {
					if( ( mi->x - start->x ) < diff ) {
						diff = mi->x - start->x;
						found = mi;
					}
				}
			}
			break;
		case KB_ESC_LEFT:
			if( mi->y == start->y ) {
				if( mi->x < start->x ) {
					if( ( start->x - mi->x ) < diff ) {
						diff = start->x - mi->x;
						found = mi;
					}
				}
			}
			break;
		default:
			return NULL;
		}
		mi = mi->next;
	}
	return found;
}

/*** process menue ***/
int proc_menu( Menu * me, char quitchar )
{
	Menuitem *	mi;
	int		ch;
	int		esc_seq;
	
	ch = '\0';
	esc_seq = 0;
	while( toupper( (char) ch ) != toupper( quitchar ) ) {
		if( ( ch = getch() ) != -1 ) {
			switch( esc_seq ) {
			case 1:
				if( ch == KB_ESC ) return 0;	/* always leave on double esc */
				if( ch == KB_ESC2 ) esc_seq = 2;
				else esc_seq = 0;
				ch = 0;
				break;
			case 2:
				if( ( mi = find_menuitem_dir( me->cur_mi, ch ) ) != NULL ) {
					me->cur_mi = mi;
					show_menu( me );
				}
				esc_seq = 0;
				ch = 0;
				break;
			default:
				if( ch == KB_ESC ) esc_seq = 1;
				else {
					esc_seq = 0;
					if( ( mi = find_menuitem_key( me->cur_mi, (char) ch ) ) != NULL ) {
						me->cur_mi = mi;
						show_menu( me );
						/* no processing of quitchar */
						if( ( toupper( ( char ) ch ) != toupper( quitchar ) ) && ( mi->func != NULL ) ) {
							/* end processing, if function returns zero */
							if( mi->func( mi->funcdata ) == 0 ) return  0;
						}
					}
					else {
						if( ch == 10 ) {
							/* no processing, if menuitems key is quitchar */
							if( toupper( me->cur_mi->key ) != toupper( quitchar ) ) {
								if( me->cur_mi->func != NULL ) {
									/* end processing, if function returns zero */
									if( me->cur_mi->func( me->cur_mi->funcdata ) == 0 ) return 0;
								}
							}
							else return;
						}
					}
				}
				break;
			}
		}
		else sched_yield();
	}
	return 1;	/* only when leaving with quitchar */
}
