/* param.c - get/set scc-driver parameters - Guido ten Dolle, PE1NNZ */

#include <stdio.h>
#include <fcntl.h>
#include <termios.h>
#include <string.h>
#include <linux/scc.h>
#define TPS             25  	/* scc_timer ticks per second */


#include <linux/fs.h>
#include <linux/tty.h>

static int speed_table[] = {
#ifdef B0
	0, B0,
#endif
#ifdef B50
	50, B50,
#endif
#ifdef B75
	75, B75,
#endif
#ifdef B110
	110, B110,
#endif
#ifdef B134
	134, B134,
#endif
#ifdef B150
	150, B150,
#endif
#ifdef B200
	200, B200,
#endif
#ifdef B300
	300, B300,
#endif
#ifdef B600
	600, B600,
#endif
#ifdef B1200
	1200, B1200,
#endif
#ifdef B1800
	1800, B1800,
#endif
#ifdef B2400
	2400, B2400,
#endif
#ifdef B4800
	4800, B4800,
#endif
#ifdef B9600
	9600, B9600,
#endif
#ifdef B19200
	19200, B19200,
#endif
#ifdef B38400
	38400, B38400,
#endif
#ifdef B57600
	57600, B57600,
#endif
#ifdef B115200
	115200, B115200,
#endif
	-1, -1
};

#define TICKS (100/TPS)
#define VAL (val=val*TPS/100)? val:1

static int
find_speed(speed)
long speed;
{
	int i;

	i = 0;
	while (speed_table[i] < speed && speed_table[i+1] > -1)
		i+=2;
	return speed_table[i+1];
}

int atonum(char *s)
{
	int n;

	while (*s == ' ')
		s++;
	if (strncmp(s, "0x", 2) == 0 || strncmp(s, "0X", 2) == 0)
		sscanf(s + 2, "%x", &n);
	else if (s[0] == '0' && s[1])
		sscanf(s + 1, "%o", &n);
	else
		sscanf(s, "%d", &n);
	return n;
}

void get_scc(int fd)
{
	struct scca a;
	struct termios t;
	char   txs[8];

	if (ioctl(fd, TIOCGSCC, &a) < 0) {
/*		perror("ERROR: Cannot get scc info"); */
		fprintf(stderr,"ERROR: this channel is not attached.\n");
		exit(1);
	}
	if(ioctl(fd, TCGETS, &t) < 0){
		perror("ERROR: ioctl failed on device");
		exit(1);
	}
	printf("\nParameters:\n\n");
	printf("speed       : %u baud\n", speed_table[(t.c_cflag & CBAUD) * 2]);
	printf("txdelay     : %u\n", a.txdelay*TICKS); 
	printf("persist     : %u\n", a.persist);
	printf("slot        : %u\n", a.slottime*TICKS);
	printf("tail        : %u\n", a.tailtime*TICKS);
	printf("fulldup     : %u\n", a.fulldup);
	printf("wait        : %u\n", a.waittime*TICKS);
	printf("maxkey      : %u sec\n", a.maxkeyup);
	printf("min         : %u sec\n", a.mintime);
	printf("idle        : %u sec\n", a.idletime);
	printf("group	    : 0x%x\n", a.group);
	printf("txoff       : %u\n", a.tx_inhibit);
	printf("maxdefer    : %u\n", a.maxdefer);

	switch(a.tstate)
	{
		case	IDLE:
			strcpy(txs,"   idle");
			break;
		case	DEFER:
			strcpy(txs,"   defer");
			break;
		case	KEYUP:
			strcpy(txs,"   keyup");
			break;
		case	KEYWT:
			strcpy(txs,"   keywt");
			break;
		case	ACTIVE:
			strcpy(txs,"  active");
			break;
		case	FLUSH:
			strcpy(txs,"   flush");
			break;
		case	TAIL:
			strcpy(txs,"    tail");
			break;
		default:
			strcpy(txs,"       ?");
			break;
	}

        printf("\nStatus:\n\n");
        printf("HDLC                     Z8530               Interrupts\n");
        printf("--------------------------------------------------------------\n");
        printf("Sent       : %7lu     Space   : %5u     RxInts : %8lu\n", a.enqueued, a.nospace, a.rxints);
        printf("Received   : %7lu     RxOver  : %5u     TxInts : %8lu\n", a.rxframes, a.rovers, a.txints);
        printf("RxErrors   : %7lu     TxOver  : %5u     ExInts : %8lu\n", a.rxerrs, a.tovers, a.exints);
	printf("KissErrors : %7lu                         SpInts : %8lu\n", a.txerrs, a.spints);
	printf("Tx State   : %s\n\n", txs);
}

void set_scc(int fd, char *cmd, unsigned int val)
{
	struct scca a;
	struct termios t;

	if (ioctl(fd, TIOCGSCC, &a) < 0) {
/*		perror("ERROR: Cannot get scc parameters"); */
		fprintf(stderr,"ERROR: this channel is not attached.\n");
		exit(1);
	}
	
	if(!(strcmp(cmd,"txdelay")))
		a.txdelay = VAL;
	if(!(strcmp(cmd,"persist")))
		a.persist = val;
	if(!(strcmp(cmd,"slot")))
		a.slottime = VAL;
	if(!(strcmp(cmd,"tail")))
		a.tailtime = VAL;
	if(!(strcmp(cmd,"fulldup")))
		a.fulldup = val;
	if(!(strcmp(cmd,"wait")))
		a.waittime = VAL;
	if(!(strcmp(cmd,"maxkey")))
		a.maxkeyup = val;
	if(!(strcmp(cmd,"min")))
		a.mintime = val;
	if(!(strcmp(cmd,"idle")))
		a.idletime = val;
	if(!(strcmp(cmd,"group")))
		a.group = val;
	if(!(strcmp(cmd,"txoff")))
		a.tx_inhibit = val;
	if(!(strcmp(cmd,"speed"))){
		if(ioctl(fd, TCGETS, &t) < 0){
			perror("ERROR: ioctl failed on device");
			exit(1);
		}
	
		t.c_iflag = (IGNBRK | IGNPAR);
		t.c_oflag = 0;
		t.c_lflag = 0;
		t.c_cflag = (find_speed(val) | CS8 | CREAD);
		t.c_line = 1;  /* special SLIP line discipline */
		t.c_cc[VEOL] = 0300;
		t.c_cc[VTIME] = 0;
		t.c_cc[VMIN] = 0;

		if (ioctl(fd, TCSETSF, &t) < 0) {
			perror("ERROR: ioctl could not set parameters");
			exit(1);
		}
	}

	if (ioctl(fd, TIOCSSCC, &a) < 0) {
		perror("ERROR: Cannot set scc parameters");
		exit(1);
	}
}

main(int argc, char **argv)
{
	int fd;

	if (argc == 1){
		fprintf(stderr,
			"usage: %s   dev [cmd,arg]\n\n", argv[0]);
		fprintf(stderr, "example: %s /dev/sc4 group 0x301\n\n",
			argv[0]);
		exit(1);
	}
	if ((fd = open(argv[1], O_RDWR)) < 0){
		perror("ERROR: Cannot open scc device");
		exit(1);
	}
	else
		if (argc == 4)
			set_scc(fd, argv[2], atonum(argv[3]));
		else
			get_scc(fd);
	exit(0);
}

