#define SCC 1
#ifdef SCC

#define BANNER "Z8530 SCC driver v1.5ALPHA! "

/************************************************************************/
/* SCC.C								*/ 
/************************************************************************/

/*
   Well... We will drop the usual history list. Among the people working
   on this driver were:
   
   PE1CHL Rob   - the original author - NET version
   KA9Q   Phil  - NOS version
   KY3B   Ken   - various changes
   DG0FT  Rene  - BayCom USCC support (for NOS)
   
   Linux version:

   PA3AOU Harry - testing, information supply and supporting
   PE1NNZ Guido - ported to Linux, ESCC support 
   DL1BKE Joerg - KISS-parameter setting and BayCom USCC support
   Hans  Alblas - memory leak ,main memory fragmentation problem
                  speed up the interupt routines ,has now its own
                  memory buffer pool and source cleanup.
*/

/* exports: scc_init(), scc_open(), scc_write() */
/* changes: in tty.h, tty_io.c and Makefile */
/* to do  : change scc.h to your needs */


#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/tqueue.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/major.h>
#include <linux/termios.h>
#include <linux/serial.h>
#include <linux/interrupt.h>
#include <linux/config.h>
#include <linux/string.h>
#include <linux/fcntl.h>
#include <linux/ptrace.h>
#include <linux/malloc.h>
#include <linux/scc.h>
#include "scc_config.h"

#include <asm/system.h>
#include <asm/io.h>
#include <asm/segment.h>
#include <asm/bitops.h>

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>
#include <linux/kernel.h>


long scc_init(long kmem_start);

int scc_open(struct tty_struct *tty, struct file *filp);
static void scc_close(struct tty_struct *tty, struct file *filp);
int scc_write(struct tty_struct *tty, int from_user, unsigned char *buf, int count);
static void scc_put_char(struct tty_struct *tty, unsigned char ch);
static void scc_flush_chars(struct tty_struct *tty);
static int scc_write_room(struct tty_struct *tty);
static int scc_chars_in_buffer(struct tty_struct *tty);
static void scc_flush_buffer(struct tty_struct *tty);
static int scc_ioctl(struct tty_struct *tty, struct file *file, unsigned int cmd, unsigned long arg);
static void scc_set_termios(struct tty_struct *tty, struct termios *old_termios);
static void scc_throttle(struct tty_struct *tty);
static void scc_unthrottle(struct tty_struct *tty);
static void scc_start(struct tty_struct *tty);
static void scc_stop(struct tty_struct *tty);

static int z8530_init(int nchips,ioaddr iobase,int space,int aoff,
                    int boff,int doff,ioaddr intack,int ivec,
                    long clk,int pclk,int hwtype,int hwparam);

static void scc_change_speed(struct sccchan *scc);


static void scc_sdlc(struct sccchan *scc);
static void scc_txon(struct sccchan *scc);
static void scc_txoff(struct sccchan *scc);
static unsigned int scc_speed (struct sccchan *scc, unsigned int clkmode,long speed);
static void scc_sdlctx(register struct sccchan *scc);
static void scc_sdlcex(register struct sccchan *scc);
static void scc_sdlcrx(register struct sccchan *scc);
static void scc_sdlcsp(register struct sccchan *scc);
static void sccvec(int irq);
static void sccnovec(int irq);
static void scc_timer(void);

/* from serial.c */

static int baud_table[] = {
	0, 50, 75, 110, 134, 150, 200, 300, 600, 1200, 1800, 2400, 4800,
	9600, 19200, 38400, 57600, 115200, 0 };


struct tty_driver scc_driver;		    /* new in 1.1.xx */
static int scc_refcount;
static struct tty_struct *scc_table[2*MAXSCC];
/*static struct termios scc_termios[2 * MAXSCC];
static struct termios scc_termios_locked[2 * MAXSCC];*/

	
struct sccinfo Sccinfo = {0};               /* global info about SCCs */
struct sccchan *Sccchan[2 * MAXSCC] = {0};  /* information per channel */
ioaddr Sccvecloc = {0};                     /* location to access for SCC vector */
unsigned char Sccmaxvec = {0};              /* maximum legal vector from SCC */
ioaddr Sccpolltab[MAXSCC+1][2] = {0,0};     /* polling table when no vectoring */


struct sccbuf{
	struct mbuf *bp;
	int inuse;
};
static struct sccbuf *sccfreelist[25] = {0};
#define TRUE	1 
#define FALSE	0
static int first_open = TRUE ;

static unsigned scc_delay(unsigned v);

static unsigned scc_delay(unsigned v)    /* delay for about 5 PCLK cycles */
/* unsigned v;                              pass-through used for input */
{
   register int i,j;			/* it takes time to save them */
   	i = j;				/* to keep the compiler happy pe1ayx */
   	return v;			/* return the passed parameter */
}


static inline int scc_paranoia_check(struct sccchan *scc, dev_t device, const char *routine)
{
#ifdef SCC_PARANOIA_CHECK
	static const char *badmagic =
		"Warning: bad magic number for Z8530 SCC struct (%d, %d) in %s\n";
        static const char *badinfo =
                "Warning: null Sccchan struct for (%d, %d) in %s\n";
       
	if (!scc) 
	{
        	printk(badinfo, MAJOR(device), MINOR(device), routine);
                return 1;
        }
        if (scc->magic != SCC_MAGIC) {
        	printk(badmagic, MAJOR(device), MINOR(device), routine);
                return 1;
        }
#endif

#ifdef SCC_DEBUG
printk("\n%s() called\n",routine);  
#endif
	return 0;
}
                                                                                                                                                                                                 

unsigned char Random = 0;		/* random number for p-persist */



/* Allocate mbuf */
struct mbuf *
alloc_mbuf(register int size)
{
int i;

if(first_open){
   first_open  = FALSE;
   for(i = 0 ; i < 25 ; i++){
	sccfreelist[i] = (struct sccbuf *)kmalloc(sizeof(struct sccbuf), GFP_KERNEL );
	sccfreelist[i]->bp = (struct mbuf *)kmalloc(sizeof(struct mbuf) , GFP_KERNEL );
	memset(sccfreelist[i]->bp ,0,sizeof(struct mbuf));
	sccfreelist[i]->inuse = 0;
	sccfreelist[i]->bp->refcnt = 0;
	sccfreelist[i]->bp->size = 1024;
	}
}


for(i = 0 ; i < 25 ; i++){
	if(sccfreelist[i]->inuse == 0){
		sccfreelist[i]->inuse = 1;
		sccfreelist[i]->bp->next = NULLBUF;
		sccfreelist[i]->bp->anext = NULLBUF;
		sccfreelist[i]->bp->dup = NULLBUF;
		sccfreelist[i]->bp->size = 1024;
		sccfreelist[i]->bp->refcnt = 1;
		sccfreelist[i]->bp->cnt = 0;
		sccfreelist[i]->bp->in_use = 0;
		return sccfreelist[i]->bp;
		}
	}
printk("SCC alloc_buf: have No free buffer\n");
return NULLBUF;
}


/* Decrement the reference pointer in an mbuf. If it goes to zero,
 * free all resources associated with mbuf.
 * Return pointer to next mbuf in packet chain
 */
struct mbuf *
free_mbuf(register struct mbuf *bp)
{
	struct mbuf *bpnext;
	int i;
	
	if(bp == NULLBUF)
		return NULLBUF;

	bpnext = bp->next;
	if(bp->dup != NULLBUF){
		for(i = 0 ; i < 25 ; i++){
			if(sccfreelist[i]->bp == bp->dup){
			      sccfreelist[i]->bp->cnt = 0;
			      sccfreelist[i]->bp->refcnt = 0;
			      sccfreelist[i]->inuse = 0;
			      bp->dup = NULLBUF;
			      }
			}
	}
	/* Decrement reference count. If it has gone to zero, free it. */
	if(--bp->refcnt <= 0){
		for(i = 0 ; i < 25 ; i++){
			if(sccfreelist[i]->bp == bp){
			      sccfreelist[i]->bp->cnt = 0;
			      sccfreelist[i]->bp->refcnt = 0;
			      sccfreelist[i]->inuse = 0;
			      return bpnext;
			      }
			}
		}
	printk("SCC free_mbuf error %p\n",bp);
	return bpnext;
}



/* Free packet (a chain of mbufs). Return pointer to next packet on queue,
 * if any
 */
struct mbuf *
free_p(register struct mbuf *bp)
{
	register struct mbuf *abp;

	if(bp == NULLBUF)
		return NULLBUF;
	abp = bp->anext;
	while(bp != NULLBUF)
		bp = free_mbuf(bp);
	return abp;
}               

/* Append packet to end of packet queue */
void
enqueue(struct mbuf **q,struct mbuf *bp)
{
	register struct mbuf *p;
	unsigned long flags;

	if(q == NULLBUFP || bp == NULLBUF)
		return;
	save_flags(flags);cli();
	
	if(*q == NULLBUF){
		/* List is empty, stick at front */
		*q = bp;
	} else {
		for(p = *q ; p->anext != NULLBUF ; p = p->anext)
			;
		p->anext = bp;
	}
	restore_flags(flags);
}

/* Append mbuf to end of mbuf chain */
void
append(struct mbuf **bph,struct mbuf *bp)
{
	register struct mbuf *p;
	unsigned long flags;

	if(bph == NULLBUFP || bp == NULLBUF)
		return;
	
	save_flags(flags);cli();
	
	if(*bph == NULLBUF){
		/* First one on chain */
		*bph = bp;
	} else {
		for(p = *bph ; p->next != NULLBUF ; p = p->next)
			;
		p->next = bp;
	}
	restore_flags(flags);
}




/* here for historic reasons */

static int
z8530_init(int nchips,ioaddr iobase,int space,int aoff,int boff,int doff,ioaddr intack,int ivec,long clk,int pclk,int hwtype,int hwparam)
/* int nchips;			number of chips */
/* ioaddr iobase;      		base of first chip */
/* int space,aoff,boff,doff; */
/* ioaddr intack;      		INTACK ioport or 0 for no INTACK */
/* int ivec;			interrupt vector number */
/* long clk;		        clock frequency */
/* int pclk;			PCLK or RTxC for clock */
/* int hwtype;			selection of special hardware types */
/* int hwparam;			extra parameter for special hardware */
{
	struct sigaction	sa;
	int chip,chan;
	ioaddr chipbase;
	register ioaddr ctrl;
    	int d;
    	int dum = 1;
    	unsigned long flags;


	Sccinfo.nchips = nchips;
	Sccinfo.maxchan = (2 * nchips) - 1;
	Sccinfo.iobase = iobase;
	Sccinfo.space = space;
	Sccinfo.off[0] = aoff;
	Sccinfo.off[1] = boff;
	Sccinfo.doff = doff;
	Sccinfo.ivec = ivec;
	Sccinfo.clk = clk;
	Sccinfo.pclk = pclk;
	Sccinfo.hwtype = hwtype;
	Sccinfo.hwparam = hwparam;

	

#define z 0

	/* reset and pre-init all chips in the system */
	for(chip = 0; chip < nchips; chip++){
		chipbase = iobase + chip * space;
		ctrl = chipbase + Sccinfo.off[0];
		save_flags(flags);cli();
		VOID(RDREG(ctrl));		/* make sure pointer is written */
		WRSCC(ctrl,R9,FHWRES);		/* force hardware reset */
		for (d = 0; d < 1000; d++)	/* wait a while to be sure */
			dum *= 10;
		for(chan = 0; chan < 2; chan++){
			Sccchan[2 * chip + chan] = NULLCHAN;
			ctrl = chipbase + Sccinfo.off[chan];

			/* initialize a single channel to no-op */
			VOID(RDREG(ctrl));		/* make sure pointer is written */
			WRSCC(ctrl,R4,z);		/* no mode selected yet */
			WRSCC(ctrl,R1,z);		/* no W/REQ operation */
			WRSCC(ctrl,R2,16 * chip);	/* chip# in upper 4 bits of vector */
			WRSCC(ctrl,R3,z);		/* disable rx */
			WRSCC(ctrl,R5,z);		/* disable tx */
			WRSCC(ctrl,R9,VIS);		/* vector includes status, MIE off */
			Sccpolltab[chip][chan] = ctrl;	/* store ctrl addr for polling */
		}

		/* Other SCC cards */

		if(hwtype & HWEAGLE)			/* this is an EAGLE card */
			WRREG(chipbase + 4,0x08);	/* enable interrupt on the board */

		if(hwtype & HWPC100)			/* this is a PC100 card */
			WRREG(chipbase,hwparam);	/* set the MODEM mode (22H normally) */

		if(hwtype & HWPRIMUS)			/* this is a PRIMUS-PC */
			WRREG(chipbase + 4,hwparam); 	/* set the MODEM mode (02H normally) */

		if (hwtype & HWDRSI) {			/* this is a DRSI PC*Packet card */
			ioaddr z8536 = chipbase + 7; 	/* point to 8536 master ctrl reg */

			/* Initialize 8536 to perform its divide-by-32 function */
			/* This part copied from N6TTO DRSI-driver */

			/* Start by forcing chip into known state */

			VOID(RDREG(z8536));		/* make sure pointer is written */
			WRSCC(z8536,CIO_MICR,0x01); 	/* force hardware reset */

			for (d = 0; d < 1000; d++)	/* wait a while to be sure */
				dum *= 10;

			WRSCC(z8536,CIO_MICR,0x00); 	/* Clear reset and start */

			/* Wait for chip to come ready */

			while (RDSCC(z8536,CIO_MICR) != 0x02)
				dum *= 10;

			WRSCC(z8536,CIO_MICR,0x26); /* NV|CT_VIS|RJA */
			WRSCC(z8536,CIO_MCCR,0xf4); /* PBE|CT1E|CT2E|CT3E|PAE */

			WRSCC(z8536,CIO_CTMS1,0xe2);/* Continuous, EOE, ECE, Pulse output */
			WRSCC(z8536,CIO_CTMS2,0xe2);/* Continuous, EOE, ECE, Pulse output */
			WRSCC(z8536,CIO_CT1MSB,0x00); /* Load time constant CTC #1 */
			WRSCC(z8536,CIO_CT1LSB,0x10);
			WRSCC(z8536,CIO_CT2MSB,0x00); /* Load time constant CTC #2 */
			WRSCC(z8536,CIO_CT2LSB,0x10);

			WRSCC(z8536,CIO_IVR,0x06);

			/* Set port direction bits in port A and B		     */
			/* Data is input on bits d1 and d5, output on d0 and d4. */
			/* The direction is set by 1 for input and 0 for output  */

			WRSCC(z8536,CIO_PDCA,0x22);
			WRSCC(z8536,CIO_PDCB,0x22);

			WRSCC(z8536,CIO_CSR1,CIO_GCB|CIO_TCB); /* Start CTC #1 running */
			WRSCC(z8536,CIO_CSR2,CIO_GCB|CIO_TCB); /* Start CTC #2 running */
		}
        	restore_flags(flags);
        }
	Sccpolltab[chip][0] = 0;        /* terminate the polling table */
	Sccvecloc = intack;             /* location of INTACK/vector read */
	Sccmaxvec = 16 * nchips;        /* upper limit on valid vector */
	/* save original interrupt vector */

	/* set interrupt vector to INTACK-generating routine  */

	if(ivec == 2)
		ivec = 9;
	if(intack)
		sa.sa_handler = sccvec;
	else
		sa.sa_handler = sccnovec;
		
	sa.sa_flags = (SA_INTERRUPT);
	sa.sa_mask = 0;
	sa.sa_restorer = NULL;
	irqaction(ivec,&sa);

	printk("channels=%u, base=0x%x, irq=%u\n",Sccinfo.nchips*2,Sccinfo.iobase,Sccinfo.ivec);

	return 0;
}


/* ----> this one is called whenever you open the device <---- */
int scc_open(struct tty_struct *tty, struct file * filp)
{
	struct sccchan *scc;
	int chan;
	
#ifdef SCC_DEBUG
printk("scc_open() called\n");
#endif

        chan = MINOR(tty->device) - tty->driver.minor_start;
	
        if ((chan < 0) || (chan > Sccinfo.maxchan))
                return -ENODEV;
        
	if(Sccchan[chan] != NULLCHAN)
	{
		scc = Sccchan[chan];
    		if (scc_paranoia_check(scc, tty->device, "scc_open"))
	                return -ENODEV;
		scc->tty_opened++;
		return 0;
	}
 
	tty->termios->c_cflag &= ~CBAUD;

	scc = (struct sccchan *) kmalloc(sizeof(struct sccchan), GFP_KERNEL);
	if (scc == NULL) return -ENOMEM;
	

	memset((char *)scc,0,sizeof(struct sccchan));
	
 	scc->tty = tty;

	scc->ctrl = Sccinfo.iobase + (chan / 2) * Sccinfo.space + Sccinfo.off[chan % 2];
	scc->data = scc->ctrl + Sccinfo.doff;

	/* configure this in scc_config.h */
	scc->fulldup = SCC_Fulldup[chan];
	scc->extclock = SCC_Extclk[chan];
	scc->nrz = SCC_NRZ[chan];
		

	/* Enhanced SCC support */
	scc->enhanced = SCC_Enhanced[chan];

	scc_sdlc(scc);              /* init SCC in SDLC mode */

	scc->speed = Sccinfo.clk / (64L * (scc_speed(scc,32,1200) + 2));/* init SCC speed */
	scc->magic = SCC_MAGIC;

	scc->a.bufsiz = BUFSIZE;        /* packet buffer size */

	/* default KISS Params */
	scc->a.txdelay = 36*TPS/100;    /* 360 ms */
	scc->a.persist = 64;            /* 25% persistence */			/* was 25 */
	scc->a.slottime = 16*TPS/100;   /* 160 ms */
#if TPS > 67
	scc->a.tailtime = 3*TPS/100;    /* 30 ms */
#else
	scc->a.tailtime = 2;            /* minimal reasonable value */
#endif
	scc->a.fulldup = 0;             /* CSMA */
	scc->a.waittime = 50*TPS/100;   /* 500 ms */
	scc->a.maxkeyup = 7;            /* 7 s */
	scc->a.mintime = 3;             /* 3 s */
	scc->a.idletime = 120;          /* 30 s */

	scc->a.maxdefer = TPS * scc->a.idletime / scc->a.slottime;
	if (scc->a.maxdefer == 0)
		scc->a.maxdefer = 400;
		
	scc->a.kiss_state = KISS_IDLE;	/* don't change this... */

	Sccchan[chan] = scc;		/* put addr in table for interrupts */
	tty->driver_data = scc;

	tty_insert_flip_char(tty, FEND, 0);
	tty_insert_flip_char(tty, 0, 0);
	queue_task(&tty->flip.tqueue, &tq_timer);

	timer_table[SCC_TIMER].fn = scc_timer;
	timer_table[SCC_TIMER].expires = 0;
	timer_active |= 1 << SCC_TIMER;

	return 0;
}


/* ----> and this whenever you close the device <---- */

static void
scc_close(struct tty_struct *tty, struct file * filp)
{
	struct sccchan *scc = tty->driver_data;
	int chan;
	unsigned long flags;

        if (!scc || scc_paranoia_check(scc, tty->device, "scc_close"))
                return;
	
	chan = MINOR(tty->device) - tty->driver.minor_start;
	
        if ((chan < 0) || (chan > Sccinfo.maxchan))
                return;

		
	if(scc->tty_opened){
		scc->tty_opened--;
		return;
	}
	
	tty->driver_data = NULL;
	save_flags(flags);cli();
	
	VOID(RDREG(scc->ctrl));		/* Make sure pointer is written */

	/* wr(scc,R1, 0);pe1ayx */	/* disable interrupts */
	/* wr(scc,R3, 0);pe1ayx */	/* disable rx */
		
/* hmm... do this and you`ll have to re-initialize the chip... :-( */
#ifdef 0	
	wr(scc,R9,(chan % 2) ? CHRB : CHRA);	/* reset the channel */
#endif
	Sccchan[chan] = NULLCHAN;

	restore_flags(flags);
	tty->stopped = 0;		
	kfree(scc);
}

/* ----> a subroutine for historic reasons <---- */

/* initialize an SCC channel in SDLC mode */
static void
scc_sdlc(register struct sccchan *scc)
{
	unsigned long flags;

    	/* set interrupt handlers */
    	scc->int_transmit = scc_sdlctx;
	scc->int_extstat = scc_sdlcex;
	scc->int_receive = scc_sdlcrx;
	scc->int_special = scc_sdlcsp;

	save_flags(flags);cli();

	wr(scc,R4,X1CLK|SDLC);		/* *1 clock, SDLC mode */
	wr(scc,R1,z);			/* no W/REQ operation */
	wr(scc,R3,Rx8|RxCRC_ENAB);	/* RX 8 bits/char, CRC, disabled */
	wr(scc,R5,Tx8|DTR|TxCRC_ENAB);	/* TX 8 bits/char, disabled, DTR */
	wr(scc,R6,z);			/* SDLC address zero (not used) */
	wr(scc,R7,FLAG);		/* SDLC flag value */
	wr(scc,R9,VIS);			/* vector includes status */
	
	if (scc->extclock)			/* when using external clocks */
	{
		/* tx clock on TCRTxCP for BayCom */

		if (Sccinfo.hwtype & HWBAYCOM)
		{
			wr(scc,R11,RCTRxCP|TCRTxCP);	/* RXclk, TRxC, TXclk RTxC */
		} else {
			wr(scc,R11,RCRTxCP|TCTRxCP);	/* RXclk RTxC, TXclk TRxC. */   
		}

		
		if (scc->nrz)			/* DF9IC modem? */
		{
			wr(scc,R10,CRCPS|NRZ|ABUNDER);
		} else {
			wr(scc,R11,CRCPS|NRZI|ABUNDER);
		}

		
		wr(scc,R14,z);			/* No BRG options */
		WRSCC(scc->ctrl,R14,DISDPLL|scc->wreg[R14]); /* No DPLL operation */
	} else {
		wr(scc,R10,CRCPS|NRZI|ABUNDER);
		if(scc->fulldup){		/* when external clock divider */
			if(Sccinfo.pclk){	/* when using PCLK as clock source */
			
/* use DPLL instead of baudrategenerator for BayCom -- why? */

				if (Sccinfo.hwtype & HWBAYCOM){
					/* RXclk DPLL, TXclk RTxC, out=DPLL (Echo duplex) */
					wr(scc,R11,RCDPLL|TCRTxCP|TRxCOI|TRxCDP);
				} else {
					/* RXclk DPLL, TXclk RTxC, out=BRG.	 external /32 TRxC->RTxC */
					wr(scc,R11,RCDPLL|TCRTxCP|TRxCOI|TRxCBR);
				}
/* --- */
				
				
			} else {
				/* RXclk DPLL, TXclk TRxC.	external TX clock to TRxC */
				wr(scc,R11,RCDPLL|TCTRxCP);
			}
		} else {			/* only half-duplex operation */
			/* RXclk DPLL, TXclk BRG. BRG reprogrammed at every TX/RX switch */
#ifdef	notdef	/* KA9Q - for PSK modem */
			wr(scc,R11,RCDPLL|TCBR);
#else
			/* DPLL -> Rx clk, DPLL -> Tx CLK, TxCLK -> TRxC pin */
			wr(scc,R11,RCDPLL|TCDPLL|TRxCOI|TRxCDP);
#endif
		}
		
		wr(scc,R14,Sccinfo.pclk? BRSRC:z);	/* BRG source = PCLK/RTxC */
		WRSCC(scc->ctrl,R14,SSBR|scc->wreg[R14]); /* DPLL source = BRG */
		WRSCC(scc->ctrl,R14,SNRZI|scc->wreg[R14]); /* DPLL NRZI mode */
	}
	
	/* enable CTS (not for Baycom), ABORT & DCD interrupts */
	wr(scc,R15,((Sccinfo.hwtype & HWBAYCOM) ? 0 : CTSIE)|BRKIE|DCDIE);

	if(scc->enhanced){
		or(scc,R15,SHDLCE|FIFOE);	/* enable FIFO, SDLC/HDLC Enhancements (From now R7 is R7') */
		wr(scc,R7,AUTOEOM);
	}

	if(RDREG(scc->ctrl) & DCD){	/* DCD is now ON */
		if (!scc->extclock)
			WRSCC(scc->ctrl,R14,SEARCH|scc->wreg[R14]); /* DPLL: enter search mode */
		or(scc,R3,ENT_HM|RxENABLE);	/* enable the receiver, hunt mode */
	}
	WRREG(scc->ctrl,RES_EXT_INT);	/* reset ext/status interrupts */
	WRREG(scc->ctrl,RES_EXT_INT);	/* must be done twice */
	scc->status = RDREG(scc->ctrl);	/* read initial status */

	or(scc,R1,INT_ALL_Rx|TxINT_ENAB|EXT_INT_ENAB); /* enable interrupts */
	or(scc,R9,MIE);			/* master interrupt enable */

	restore_flags(flags);
}

/***************************************************************************/


/* ----> set SCC channel speed <---- */

/* clkmode specifies the division rate (1,16,32) inside the SCC
 * returns the selected brgrate for "real speed" calculation
 */
static unsigned int
scc_speed(register struct sccchan *scc,unsigned int clkmode,long speed)
{
	unsigned int brgrate;
	long spdclkm;
	unsigned long flags;

	/* calculate baudrate generator value */

 	if ((spdclkm = speed * clkmode) == 0)
 	return 65000U;              	/* avoid divide-by-zero */

	brgrate = (unsigned) ((Sccinfo.clk + spdclkm) / (spdclkm * 2)) - 2;

 	save_flags(flags);cli();

	cl(scc,R14,BRENABL);		/* disable baudrate generator */
	wr(scc,R12,brgrate);		/* brg rate LOW */
	wr(scc,R13,brgrate >> 8);   	/* brg rate HIGH */
	or(scc,R14,BRENABL);		/* enable baudrate generator */

	restore_flags(flags);

	return brgrate;
}

/*
 * Change scc_speed
 */
static void
scc_change_speed(struct sccchan * scc)
{
	if (scc == NULLCHAN)
		return;
	scc->speed = Sccinfo.clk / (64L * (scc_speed(scc,32,baud_table[scc->tty->termios->c_cflag & CBAUD]) + 2));
}



/* ----> ioctl-routine of the driver <---- */

/* perform ioctl on SCC (sdlc) channel
 * this is used for AX.25 mode, and will set the "kiss" parameters
 */
 
/* TIOCMGET - get modem status
 * TIOCMBIS - set PTT
 * TIOCMBIC - reset PTT
 * TIOCMBIC - set PTT
 * TIOCGSCC - get scc parameters
 * TIOCSSCC - set scc parameters
 */
 

static int
scc_ioctl(struct tty_struct *tty, struct file * file, unsigned int cmd, unsigned long arg)
{
	struct sccchan * scc = tty->driver_data;
	unsigned int result;
	unsigned int value;
	struct scca * info;
	struct termios * t;
	int error;

	if (scc_paranoia_check(scc, tty->device, "scc_ioctl"))
        	return -ENODEV;

	value = get_fs_long((unsigned long *) arg);
	info = (struct scca *) arg;
	t = (struct termios *) arg;


	switch(cmd){
	case TCSBRK:
		return 0;
	case TIOCMGET:
		error = verify_area(VERIFY_WRITE, (void *) arg,sizeof(unsigned int *));
		if (error)
			return error;

		result =  ((scc->wreg[R5] & RTS) ? TIOCM_RTS : 0)
			| ((scc->wreg[R5] & DTR) ? TIOCM_DTR : 0)
			| ((RDREG(scc->ctrl) & DCD) ? TIOCM_CAR : 0)
			| ((RDREG(scc->ctrl) & CTS) ? TIOCM_CTS : 0);
		put_fs_long(result,(unsigned long *) arg);
		return 0;
	case TIOCMBIS:
	case TIOCMBIC:
	case TIOCMSET:
		switch (cmd) {
		case TIOCMBIS:
			scc->wreg[R5] |= DTR;
			scc->wreg[R5] |= RTS;
			break;
		case TIOCMBIC:
			scc->wreg[R5] |= ~DTR;
			scc->wreg[R5] |= ~RTS;
			break;
		case TIOCMSET:
			if(value & TIOCM_DTR)
				scc->wreg[R5] |= DTR;
			else
				scc->wreg[R5] &= ~DTR;
			if(value & TIOCM_RTS)
				scc->wreg[R5] |= RTS;
			else
				scc->wreg[R5] &= ~RTS;
			break;
		}
		
		if(scc->a.tstate == IDLE && scc->timercount == 0)
		scc->timercount = 1;	/* force an update */

		return 0;
		
	case TCGETS:
		error = verify_area(VERIFY_WRITE, (void *) arg, sizeof(struct termios));
		if (error)
			return error;
		if (!t) 
			return -EFAULT;
			
		memcpy_tofs(t, scc->tty->termios, sizeof(struct termios));
		return 0;
		
	case TCSETS:
	case TCSETSF:		/* should flush first, but... */
	case TCSETSW:		/* should wait 'till flush, but... */
		if (!suser())
			return -EPERM;
		if (!t)
			return -EFAULT;
		
		memcpy_fromfs(scc->tty->termios, t, sizeof(struct termios));
		scc_change_speed(scc);
		return 0;
		
		
	case TIOCGSCC:

		error = verify_area(VERIFY_WRITE, (void *) arg,sizeof(struct scca));
		if (error)
			return error;
		
		if (!info)
			return -EFAULT;
			
		if (!scc->tty_opened)
			return -EFAULT;
			
		memcpy_tofs(info,&scc->a,sizeof(struct scca));
		return 0;
	case TIOCSSCC:
		if (!suser())
			return -EPERM;
		if (!info)
			return -EFAULT;
			
		if (!scc->tty_opened)
			return -EFAULT;
			
		memcpy_fromfs(&scc->a,info,sizeof(struct scca));
		return 0;
	default:
		printk("scc_ioctl(): invalid command %4.4x\n", cmd);
		return -EINVAL;
    }
}


/* ----- TERMIOS function ----- */

static void
scc_set_termios(struct tty_struct * tty, struct termios * old_termios)
{
	if (tty->termios->c_cflag == old_termios->c_cflag) 
		return;
	scc_change_speed(tty->driver_data);
}



/*
 * this will set the "kiss" parameters through kiss itself
 */
 
static void
kiss_set_param(struct sccchan *scc,char cmd, unsigned int val)
{

#define VAL (val=val*TPS/100)? val:1
#define SVAL val? val:1

	switch(cmd){
	case PARAM_TXDELAY:
		scc->a.txdelay = VAL; break;
	case PARAM_PERSIST:
		scc->a.persist = val; break;
	case PARAM_SLOTTIME:
		scc->a.slottime = VAL; break;
	case PARAM_TXTAIL:
		scc->a.tailtime = VAL; break;
	case PARAM_FULLDUP:
		scc->a.fulldup = val; break;
	case PARAM_WAIT:
		scc->a.waittime = VAL; break;
	case PARAM_MAXKEY:
		scc->a.maxkeyup = SVAL; break;
	case PARAM_MIN:
		scc->a.mintime = SVAL; break;
	case PARAM_IDLE:
		scc->a.idletime = val; break;
	case PARAM_GROUP:
		scc->a.group = val;  break;
	case PARAM_TX:
		scc->a.tx_inhibit = val; 
	}
	return;
}


/* interpret frame: strip CRC and decode KISS */

static void kiss_interpret_frame(struct sccchan * scc)
{
	unsigned char *p, kisscmd;
	unsigned long flags;

	if (scc->sndq1->cnt < 2)
	{
		if (scc->sndq1) 
			free_p(scc->sndq1);
		else
			scc->sndq1 = NULLBUF;
			
		scc->sndq2 = NULLBUF;
		return;
	}
	
	p = scc->sndq1->data;
	
	kisscmd = *p;
	scc->sndq1->cnt--;
	memcpy(p, p+1, scc->sndq1->cnt); 	/* I know, I know... */	
	
	if (kisscmd & 0xa0)
	{
		if (scc->sndq1->cnt > 2)
			scc->sndq1->cnt -= 2;
		else
		{
			free_p(scc->sndq1);
			scc->sndq2 = NULLBUF;
			return;
		}
	}
	
	
	kisscmd &= 0x1f;
	
		
	if (kisscmd)
	{
		kiss_set_param(scc, kisscmd, *scc->sndq1->data);
		scc->sndq1->cnt=0;
					
		free_p(scc->sndq1);
		scc->sndq2 = NULLBUF;
		return;
	}
	
	enqueue(&scc->sndq,scc->sndq1); /* enqueue packet */
	scc->sndq2 = NULLBUF;		/* acquire a new buffer next time */
	scc->a.enqueued++;

	save_flags(flags);cli();

	if(scc->a.tstate == IDLE)
	{      				/* when transmitter is idle */
		scc->a.tstate = DEFER;  	/* start the key-up sequence */
		scc->a.maxdefer = TPS * scc->a.idletime / scc->a.slottime;
	
		if (scc->a.maxdefer == 0) 
			scc->a.maxdefer++; 	/* must not reach zero */

		scc->timercount = scc->a.waittime;
	
		restore_flags(flags);
	}
}

static void kiss_store_byte(struct sccchan *scc, unsigned char ch)
{
	if(scc->sndq2->cnt == scc->sndq2->size)		/* buffer full? */
	{
		if((scc->sndq2 = alloc_mbuf(scc->a.bufsiz)) == NULLBUF)
			return;
		append(&scc->sndq1,scc->sndq2);         /* enqueue packet */
	}
	
	scc->sndq2->data[scc->sndq2->cnt++] = ch;
}

/* ----> tx routine: decode KISS data and enqueue it <---- */

/* send raw frame to SCC. used for AX.25 */
int scc_write(struct tty_struct *tty, int from_user, unsigned char *buf, int count)
{
	struct sccchan * scc = tty->driver_data;
	unsigned char ch;
	unsigned char tbuf[BUFSIZE], *p;
	int cnt, cnt2;
	
	if (!tty) return 0;
	
	if (scc_paranoia_check(scc, tty->device, "scc_write"))
		return 0;

	if (scc->a.tx_inhibit) return count;

	cnt2 = count;
	
	while (cnt2)
	{
		cnt   = cnt2 > BUFSIZE? BUFSIZE:cnt2;
		cnt2 -= cnt;
		
		if (from_user)
			memcpy_fromfs(tbuf, buf, cnt);
		else
			memcpy(tbuf, buf, cnt);
		
		buf += cnt;
			
		p=tbuf;
		
		while(cnt)
		{
			ch = *p;
			switch (scc->a.kiss_state) 
			{
				case KISS_IDLE:
					if (ch == FEND)
					{
						if ((scc->sndq2 = scc->sndq1 = alloc_mbuf(scc->a.bufsiz)) == NULLBUF)
							return 0;
						scc->a.kiss_state = KISS_DATA;
					} else scc->a.txerrs++;
					break;
					
				case KISS_DATA:
					if (ch == FESC)
						scc->a.kiss_state = KISS_ESCAPE;
					else if (ch == FEND)
					{
						
						kiss_interpret_frame(scc);	
						scc->a.kiss_state = KISS_IDLE;
					}
					else kiss_store_byte(scc, ch);
					break;
					
				case KISS_ESCAPE:
					if (ch == TFEND)
					{
						kiss_store_byte(scc, FEND);
						scc->a.kiss_state = KISS_DATA;
					}
					else if (ch == TFESC)
					{
						kiss_store_byte(scc, FESC);
						scc->a.kiss_state = KISS_DATA;
					}
					else
					{
						free_p(scc->sndq1);
						scc->sndq2 = NULLBUF;
						scc->a.txerrs++;
						scc->a.kiss_state = KISS_IDLE;
					}
			} /* switch */
			
			p++;
			cnt--;
		}  /* while cnt */
	} /* while cnt2 */
	
	return count;
}
				


/* put a single char into the buffer */

static void scc_put_char(struct tty_struct * tty, unsigned char ch)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_put_char"))
		return;
		
	scc_write(tty, 0, &ch, 1);	/* that's all */
}

static void scc_flush_chars(struct tty_struct * tty)
{
	struct sccchan *scc = tty->driver_data;
	
	scc_paranoia_check(scc, tty->device, "scc_flush_chars"); /* just to annoy the user... */
	
	return;	/* no flush needed */
}

static int scc_write_room(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_write_room"))
		return 0;
	
	return BUFSIZE;
}

static int scc_chars_in_buffer(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc && scc->sndq2)
		return scc->sndq2->cnt;
	else
		return 0;
}

static void scc_flush_buffer(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_flush_buffer"))
		return;
		
	scc->a.kiss_state = KISS_IDLE;
	
	wake_up_interruptible(&tty->write_wait);
	if ((tty->flags & (1 << TTY_DO_WRITE_WAKEUP)) &&
	    tty->ldisc.write_wakeup)
		(tty->ldisc.write_wakeup)(tty);
}

static void scc_throttle(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_throttle"))
		return;
		
		
	/* dummy */
}

static void scc_unthrottle(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_unthrottle"))
		return;
		
	/* dummy */
}

static void scc_start(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_start"))
		return;
		
	/* dummy */
}
	                                            	

static void scc_stop(struct tty_struct *tty)
{
	struct sccchan *scc = tty->driver_data;
	
	if (scc_paranoia_check(scc, tty->device, "scc_stop"))
		return;
		
	/* dummy */
}

/* ----> Interrupt handlers for sdlc mode (AX.25) <---- */

/* Transmitter interrupt handler */
static void
scc_sdlctx(register struct sccchan *scc)
{
	register struct mbuf *bp;
	int x;

	scc->a.txints++;

	switch(scc->a.tstate){          /* look at transmitter state */
	case ACTIVE:                    /* busy sending data bytes */
		while ((bp = scc->tbp)->cnt == 0){      /* nothing left in this mbuf? */
			bp = bp->next;                  /* save link to next */
			free_mbuf(scc->tbp);    /*KM*/
			if((scc->tbp = bp) == NULLBUF){/* see if more mbufs follow */
				if(RDREG(scc->ctrl) & TxEOM){   /* check tx underrun status */
					scc->a.tovers++;          /* oops, an underrun! count them */
					WRREG(scc->ctrl,SEND_ABORT);/* send an abort to be sure */
#ifdef notdef
					scc->a.tstate = TAIL;   /* key down tx after TAILTIME */
					scc->timercount = scc->a.tailtime;
					return;
#else
					for(x=0;x!=1000;x++);
					WRREG(scc->ctrl,RES_EOM_L); /* reset the EOM latch */
					for(x=0;x!=1000;x++);
					WRREG(scc->ctrl,ERR_RES);
					for(x=0;x!=1000;x++);
					
					scc->a.tstate = FLUSH;   /* key down tx after TAILTIME */
					scc->timercount = scc->a.tailtime;
#endif
				}
				
				cl(scc,R10,ABUNDER);            /* frame complete, allow CRC transmit */
				scc->a.tstate = FLUSH;
				WRREG(scc->ctrl,RES_Tx_P);      /* reset pending int */
				return;
			}
		}
		/* now bp = scc->tbp (either from while or from if stmt above) */
		WRREG(scc->data,bp->data[bp->in_use++]); /* send the character */
		bp->cnt--;                      /* decrease mbuf byte count */
		return;
	case FLUSH:     /* CRC just went out, more to send? */
#ifdef notdef	
		or(scc,R10,ABUNDER);            /* re-install underrun protection */
#endif
		/* verify that we are not exeeding max tx time (if defined) */
		if((scc->timercount != 0 || scc->a.maxkeyup == 0) &&
		 (scc->tbp = scc->sndq) != NULLBUF){ /* dequeue a frame */
			scc->sndq = scc->sndq->anext;
			WRREG(scc->ctrl,RES_Tx_CRC); /* reset the TX CRC generator */
			scc->a.tstate = ACTIVE;
			scc_sdlctx(scc);             /* write 1st byte */
			if(!(scc->enhanced))
				WRREG(scc->ctrl,RES_EOM_L);  /* reset the EOM latch */
			return;
		}
		scc->a.tstate = TAIL;           /* no more, key down tx after TAILTIME */
		scc->timercount = scc->a.tailtime;
		WRREG(scc->ctrl,RES_Tx_P);
		return;
	default:                                /* another state */
		WRREG(scc->ctrl,RES_Tx_P);      /* then don't send anything */
		return;
	}
}





/* ----> Switch the SCC to "transmit" mode <---- */

/* Only to be called from an interrupt handler, while in AX.25 mode */
static void
scc_txon(register struct sccchan *scc)
{
	if (!scc->fulldup && !scc->extclock){ /* no fulldup divider? */
		cl(scc,R3,RxENABLE);		/* then switch off receiver */
		cl(scc,R5,TxENAB);		/* transmitter off during switch */
		scc_speed(scc,1,scc->speed);	/* reprogram baudrate generator */

/* !!!! reprogram the Baudrategenerator !!!! */	
/* !!!! DL1BKE: Hmm, I think this is necessary for any card without
                a fullduplex divider. If you run into trouble, just
                remove the #ifdef/#endif constructions. See also scc_txoff() */
#ifdef 0
		if (Sccinfo.hwtype & HWBAYCOM){
#endif		
			/* DPLL -> Rx clk, BRG -> Tx CLK, TxCLK -> TRxC pin */
			wr(scc, R11, RCDPLL|TCBR|TRxCOI|TRxCBR);
#ifdef 0			
		}
#endif

/* --- */		
			
	}
	or(scc,R5,RTS|TxENAB);			/* set the RTS line and enable TX */

	if(Sccinfo.hwtype & HWPRIMUS)		/* PRIMUS has another PTT bit ... */ 
		WRREG(scc->ctrl + 4,Sccinfo.hwparam | 0x80);	/* set that bit ! */
}


/* ---->  Switch the SCC to "receive" mode (or: switch off transmitter) <---- */

/* Only to be called from an interrupt handler, while in AX.25 mod */

static void
scc_txoff(register struct sccchan *scc)
{
	cl(scc,R5,RTS);			/* turn off RTS line */
	if(Sccinfo.hwtype & HWPRIMUS)	/* PRIMUS has another PTT bit... */
		WRREG(scc->ctrl + 4,Sccinfo.hwparam);	/* clear that bit ! */

	if (!scc->fulldup && !scc->extclock){ /* no fulldup divider? */
		cl(scc,R5,TxENAB);		/* then disable the transmitter */
		scc_speed(scc,32,scc->speed);	/* back to receiver baudrate */
		
/* !!!! dito !!!! */
/* !!!! DL1BKE: see scc_txon() */

#ifdef 0
		if (Sccinfo.hwtype & HWBAYCOM){
#endif				
			/* DPLL -> Rx clk, DPLL -> Tx CLK, RxCLK -> TRxC pin */
			wr(scc, R11, RCDPLL|TCDPLL|TRxCOI|TRxCDP); 
#ifdef 0			
		}
#endif		
/* --- */
		
	}
}


/* ----> interrupt service routines for the 8530 <---- */

/* sccvec is the interrupt handler for SCC interrupts using INTACK
 */
static void
sccvec(int irq)
{
    register unsigned char vector;
    register struct sccchan *scc;
    register void (*handler)(register struct sccchan *scc);
    unsigned long flags;

    save_flags(flags);cli();
    /* Read SCC interrupt vector and check it */
    for(;;){ 
        outb(0,Sccvecloc);      /* Generate INTACK */
  
        /* Read the vector */
        if((vector=inb(Sccvecloc)) >= Sccmaxvec)
          break;
        /* Extract channel number and status from vector. */
        /* Determine handler address. */
        /* Isolate channel nummer */
        if((vector & 0x01) || ((scc=Sccchan[(((vector>>1)&0x7c)^0x04) >> 2]) == NULLCHAN))
          break; 

        /* Isolate status info from vector, get handler, call handler */
        handler = (&scc->int_transmit)[(vector & 0x06) >> 1];
        handler(scc);                       /* Call the handler */
        outb(0x38,scc->ctrl);               /* Reset Highest IUS" opcode to WR0 */
    
    }  
    restore_flags(flags);
}

/* sccnovec is the interrupt handler for SCC interrupts using polling
 */
static void
sccnovec(int irq)
{
    register unsigned char vector;
    register struct sccchan *scc;
    register void (*handler)(register struct sccchan *scc);
    register ioaddr *p;
    register ioaddr q;

    cli();
    /* Find the SCC generating the interrupt by polling all attached SCCs
     * reading RR3A (the interrupt pending register)
     */
     
    p = (unsigned short *) Sccpolltab;
    
    while( (ioaddr) q = (ioaddr *)(p++)){             
     outb(3,q);                         /* Read A CTRL address; Select RR3A */
     
     if(inb(q)){
        outb(2,q=*p);                   /* Read B CTRL address; Select RR2B */
        vector=inb(q);                  /* Read the vector */
        
        /* Extract channel number and status from vector. */
        /* Determine handler address. */
        /* Isolate channel nummer */
        if((vector & 1) || ((scc=Sccchan[(((vector>>1)&0x7c)^0x04) >> 2]) == NULLCHAN))
          break; 

        /* Isolate status info from vector, get handler, call handler */

        handler = (&scc->int_transmit)[(vector & 0x06) >> 1];
        handler(scc);
        p = (unsigned short *) Sccpolltab;
     }
    }    
    sti();
}


/* ----> SCC timer interrupt handler. Will be called every 1/TPS s <---- */

static void
scc_timer(void)
{
	register struct sccchan *scc;
	register struct sccchan **sccp;
	unsigned long flags;

	save_flags(flags);cli();
	
	for(sccp = Sccchan + Sccinfo.maxchan; sccp >= Sccchan; sccp--){
		if((scc = *sccp) != NULLCHAN){
		  if(scc->timercount != 0 &&
		    --(scc->timercount) == 0){
			/* handle an SCC timer event for this SCC channel
			 * this can only happen when the channel is AX.25 type
			 */
			switch(scc->a.tstate){
			case IDLE:			/* it was idle, this is FULLDUP2 timeout */
				scc_txoff(scc);		/* switch-off the transmitter */
				break;
			case DEFER:			/* trying to get the channel */
				/* operation is as follows:
				 * CSMA: when channel clear AND persistence randomgenerator
				 *	 wins, AND group restrictions allow it:
				 *		keyup the transmitter
				 *	 if not, delay one SLOTTIME and try again
				 * FULL: always keyup the transmitter
				 */
				if(scc->a.fulldup == 0){
					Random = 21 * Random + 53;
					if(scc->status & DCD || scc->a.persist < Random){
						/* defer transmission again. check for limit */
defer_it:					if(--(scc->a.maxdefer) == 0){
							/* deferred too long. choice is to:
							 * - throw away pending frames, or
							 * - smash-on the transmitter and send them.
							 * the first would be the choice in a clean
							 * environment, but in the amateur radio world
							 * a distant faulty station could tie us up
							 * forever, so the second may be better...
							*/
#ifdef THROW_AWAY_AFTER_DEFER_TIMEOUT
#else
							goto keyup; /* just keyup the transmitter... */
#endif
						}
						scc->timercount = scc->a.slottime;
						break;
					}
					if(((unsigned char) scc->a.group) != NORTXGROUP){
						int i;
						struct sccchan *scc2;

						for(i = 0; i <= Sccinfo.maxchan; i++)
							if((scc2 = Sccchan[i]) != NULLCHAN &&
							 scc2 != scc &&
							 ((unsigned char) scc2->a.group) & ((unsigned char) scc->a.group) &&
							 ((scc->a.group & TXGROUP && scc2->wreg[R5] & RTS) ||
							 (scc->a.group & RXGROUP && scc2->status & DCD))){
								goto defer_it;
							}
					}
				}
			case KEYUP:			/* keyup transmitter (note fallthrough) */
keyup:				if((scc->wreg[R5] & RTS) == 0){ /* when not yet keyed */
					scc->a.tstate = KEYWT;
					scc->timercount = scc->a.txdelay; /* 0 if CTSwait */
					scc_txon(scc);
					break;
				}
				/* when already keyed, directly fall through */
			case KEYWT:                     /* waited for CTS or TXDELAY */
				/* when a frame is available (it should be...):
				 * - dequeue it from the send queue
				 * - reset the transmitter CRC generator
				 * - set a timeout on transmission length, if defined
				 * - send the first byte of the frame
				 * - reset the EOM latch
				 * when no frame available, proceed to TAIL handling
				 */
				if((scc->tbp = scc->sndq) != NULLBUF){
					scc->sndq = scc->sndq->anext;
					WRREG(scc->ctrl,RES_Tx_CRC);
					scc->a.tstate = ACTIVE;
					scc->timercount = TPS * scc->a.maxkeyup;
					scc_sdlctx(scc);
					if(!(scc->enhanced)) 
						WRREG(scc->ctrl,RES_EOM_L);
					break; 
				}
				/* when no frame queued, fall through to TAIL case */
			case TAIL:                      /* at end of frame */
				/* when fulldup is 0 or 1, switch off the transmitter.
				 * when frames are still queued (because of transmit time limit),
				 * restart the procedure to get the channel after MINTIME.
				 * when fulldup is 2, the transmitter remains keyed and we
				 * continue sending.    IDLETIME is an idle timeout in this case.
				 */     
				if(scc->a.fulldup < 2){
					scc->a.tstate = IDLE;
					scc_txoff(scc);

					if(scc->sndq != NULLBUF){
						scc->a.tstate = DEFER;
						scc->a.maxdefer = TPS * scc->a.idletime /
						 scc->a.slottime;
						if (scc->a.maxdefer == 0) scc->a.maxdefer++;
							/* dl1bke 940511: must not be zero! */
 
						scc->timercount = TPS * scc->a.mintime;
					}
					break;
				}
				if(scc->sndq != NULLBUF){ /* still frames on the queue? */
					scc->a.tstate = KEYWT; /* continue sending */
					scc->timercount = TPS * scc->a.mintime; /* after mintime */
				} else {
					scc->a.tstate = IDLE;
					scc->timercount = TPS * scc->a.idletime;
				}
				break;
			case ACTIVE:	/* max keyup time expired */
			case FLUSH:	/* same while in flush mode */
				break;	/* no action required yet */
			default:			/* unexpected state */
				scc->a.tstate = IDLE;	/* that should not happen, but... */
				scc_txoff(scc);		/* at least stop the transmitter */
				break;
			}
		  }

		}
	}
	timer_table[SCC_TIMER].fn = scc_timer;
	timer_table[SCC_TIMER].expires = jiffies + HZ/TPS;
	timer_active |= 1 << SCC_TIMER; 
	restore_flags(flags);
}


/* ----------------------------------------------------------------------- */

/* ---->  SCC driver initialisation. Called at boot time <---- */

long scc_init (long kmem_start)
{       struct termios scc_termios[2 * MAXSCC];
        struct termios scc_termios_locked[2 * MAXSCC];

       
        memset(&scc_driver, 0, sizeof(struct tty_driver));
        scc_driver.magic = TTY_DRIVER_MAGIC;
        scc_driver.name = "sc";
        scc_driver.major = TTY_MAJOR;		
        scc_driver.minor_start = 96;
        scc_driver.num = MAXSCC*2;
        scc_driver.type = TTY_DRIVER_TYPE_SERIAL;
        scc_driver.subtype = 0;			/* not needed */
        scc_driver.init_termios = tty_std_termios;
        scc_driver.init_termios.c_cflag = B9600	| CS8 | CREAD | HUPCL | CLOCAL;
        scc_driver.flags = TTY_DRIVER_REAL_RAW;
        scc_driver.refcount = &scc_refcount;	/* not needed yet */
        scc_driver.table = scc_table;
        scc_driver.termios = (struct termios **) scc_termios;
        scc_driver.termios_locked = (struct termios **) scc_termios_locked;
        scc_driver.open = scc_open;
        scc_driver.close = scc_close;
        scc_driver.write = scc_write;
        scc_driver.start = scc_start;
        scc_driver.stop = scc_stop;
        
        scc_driver.put_char = scc_put_char;
        scc_driver.flush_chars = scc_flush_chars;        
	scc_driver.write_room = scc_write_room;
	scc_driver.chars_in_buffer = scc_chars_in_buffer;
	scc_driver.flush_buffer = scc_flush_buffer;
	
	scc_driver.throttle = scc_throttle;
	scc_driver.unthrottle = scc_unthrottle;
        
        scc_driver.ioctl = scc_ioctl;
        scc_driver.set_termios = scc_set_termios;
        
        if (tty_register_driver(&scc_driver))
           panic("Couldn't register Z8530 SCC driver\n");
                                
	printk (BANNER);
	z8530_init(NCHIPS, IOBASE, SPACE, AOFF, BOFF, DOFF, INTACK, IVEC, CLK, PCLK, HWTYPE, HWPARAM); 
	return kmem_start;
}


/* External/Status interrupt handler */
static void
scc_sdlcex(register struct sccchan *scc)
{
	register unsigned char status,changes;
	unsigned long flags;

	scc->a.exints++;

	save_flags(flags);cli();

	status = RDREG(scc->ctrl);
	changes = status ^ scc->status;

	if(changes & BRK_ABRT){         /* Received an ABORT */
		if(status & BRK_ABRT){          /* is this the beginning? */
			if(scc->rxbufcnt != 0){/* did we receive something? */
				scc->rxbufcnt = 0;       /* throw away buffer */
			}
			VOID(RDREG(scc->data)); /* flush the FIFO */
			VOID(RDREG(scc->data));
			VOID(RDREG(scc->data));
		}
	}
	if(changes & CTS){                      /* CTS input changed state */
		if(status & CTS){               /* CTS is now ON */
			if(scc->a.tstate == KEYWT &&
				scc->a.txdelay == 0) /* zero TXDELAY = wait for CTS */
			scc->timercount = 1;    /* it will start within 10 ms */
		}
	}
	if(changes & DCD){                      /* DCD input changed state */
		if(status & DCD){               /* DCD is now ON */
			if (!scc->extclock)
				WRSCC(scc->ctrl,R14,SEARCH|scc->wreg[R14]); /* DPLL: enter search mode */
			or(scc,R3,ENT_HM|RxENABLE); /* enable the receiver, hunt mode */
		} else {                        /* DCD is now OFF */
			cl(scc,R3,ENT_HM|RxENABLE); /* disable the receiver */
			VOID(RDREG(scc->data)); /* flush the FIFO */
			VOID(RDREG(scc->data));
			VOID(RDREG(scc->data));
			if(scc->rxbufcnt != 0){/* did we receive something? */
				scc->rxbufcnt = 0;         /* throw away buffer */
			}
		}
	}
	scc->status = status;
	WRREG(scc->ctrl,RES_EXT_INT);
	restore_flags(flags);

}

/* Receiver interrupt handler */
static void
scc_sdlcrx(register struct sccchan *scc)
{
	unsigned long flags;


	scc->a.rxints++;

	if((scc->a.tstate == ACTIVE) && (scc->a.fulldup == 0))
	{
		VOID(RDREG(scc->data));	/* so we have to discard the char */
		or(scc,R3,ENT_HM);	/* enter hunt mode for next flag */
		return;
	}
	if(scc->rxbufcnt > 510){
		VOID(RDREG(scc->data)); /* so we have to discard the char */
		or(scc,R3,ENT_HM);      /* enter hunt mode for next flag */
		scc->rxbufcnt = 0;      /* put buffers back on pool */
		scc->a.nospace++;       /* count these events */
		return;
	}

	/* we have a bufferplace read character and store it */

	save_flags(flags);cli();
	scc->rxbuf[scc->rxbufcnt++] = RDREG(scc->data);
	restore_flags(flags);

}


/* Receive Special Condition interrupt handler */
static void
scc_sdlcsp(register struct sccchan *scc)
{
	register unsigned char status;
	struct tty_struct * tty = scc->tty;
	unsigned char ch;
	int i;


	unsigned long flags;
	save_flags(flags);cli();

	scc->a.spints++;
	status = rd(scc,R1);            /* read receiver status */
	VOID(RDREG(scc->data));         /* flush offending character */

	if(status & Rx_OVR){            /* receiver overrun */
		scc->a.rovers++;                  /* count them */
		or(scc,R3,ENT_HM);              /* enter hunt mode for next flag */
		scc->rxbufcnt = 0;              /* rewind the buffer and toss */
	}
	if(status & END_FR &&           /* end of frame */
	scc->rxbufcnt != 0){           /* at least received something */
		if((status & CRC_ERR) == 0 &&   /* no CRC error is indicated */
		(status & 0xe) == RES8 &&       /* 8 bits in last byte */
		scc->rxbufcnt > 0){
   		   /* we seem to have a good frame. but the last byte received */
		   /* from rx interrupt is in fact a CRC byte, so discard it */
		   scc->rxbufcnt--;       /* strip CRC */
		   if((tty->flip.count + scc->rxbufcnt + 70) < TTY_FLIPBUF_SIZE-1){
			tty_insert_flip_char(tty, FEND, 0);
			tty_insert_flip_char(tty, 0, 0);
			for(i = 0 ; i < scc->rxbufcnt ; i++){
			    switch(ch = scc->rxbuf[i]){
			        case FEND:
				   tty_insert_flip_char(tty, FESC, 0);
			  	   tty_insert_flip_char(tty, TFEND, 0);
				   break;
			        case FESC:
			 	   tty_insert_flip_char(tty, FESC, 0);
			  	   tty_insert_flip_char(tty, TFESC, 0);
				   break;
			        default:
				   tty_insert_flip_char(tty, ch, 0);
				}
			}							
			tty_insert_flip_char(tty, FEND, 0);
			queue_task_irq_off(&tty->flip.tqueue, &tq_timer);  
			scc->a.rxframes++; 
		   }
		   else 	scc->a.rovers++;    /* count them */
 		   scc->rxbufcnt = 0;               /* throw away frame */
		} else {                        /* a bad frame */
			scc->rxbufcnt = 0;      /* throw away frame */
			scc->a.rxerrs++;
		}
	}
	WRREG(scc->ctrl,ERR_RES);
	restore_flags(flags);
}


#endif
