#ifndef	_SCC_H
#define	_SCC_H

/* selection of hardware types */

#define HWSCC		0x00	/* hardware type for PA0HZP SCC card */
#define HWEAGLE         0x01    /* hardware type for EAGLE card */
#define HWPC100		0x02	/* hardware type for PC100 card */
#define HWPRIMUS	0x04	/* hardware type for PRIMUS-PC (DG9BL) card */
#define HWDRSI		0x08	/* hardware type for DRSI PC*Packet card */
#define HWBAYCOM	0x10	/* hardware type for BayCom (U)SCC */

/* Constants */

/* Something to put in: /usr/include/linux/timer.h ??? */
#define SCC_TIMER	3

/* Something to put in: /usr/include/linux/termios.h ??? */
#define TIOCGSCC	0x5470
#define TIOCSSCC	0x5471

/* magic number for Linux 1.1.xx and higher */

#define SCC_MAGIC	0x8530
#define SCC_PARANOIA_CHECK

/* KISS protocol flags */
#define FEND	192
#define FESC	219
#define TFEND	220
#define TFESC	221

#define SC_TTY_MASK	0x20

#define INLINE 1

typedef unsigned short ioaddr;	/* type definition for an 'io port address' */

# if defined(INLINE)
/* special delay construction only necessary when inline IN/OUT is used */
#define D(v)		scc_delay(v)	/* delay for 5 PCLK cycles (or more) */
#define RDREG(a)	(D(inb(a)))		/* read any input port */
#define WRREG(a,v)	{outb(v,a); D(1);}	/* write any output port */
#define RDSCC(c,r)	(outb(r,c), D(1), D(inb(c)))        /* read SCC reg */
#define WRSCC(c,r,v)	{outb(r,c); D(1); outb(v,c); D(1);} /* write SCC reg*/
# else
#define RDREG(a)	(inb(a))		/* read any input port */
#define WRREG(a,v)	{outb(v,a);}		/* write any output port */
#define RDSCC(c,r)	(outb(r,c), inb(c))     /* read SCC reg */
#define WRSCC(c,r,v)	{outb(r,c); outb(v,c);} /* write SCC reg */
# endif

#ifndef VOID
#define VOID(x)		(x)	/* not necessary for most compilers */
#endif

struct sccinfo {
	int nchips;		/* Number of SCC chips in system */
	int maxchan;		/* Highest valid channel number */
	ioaddr iobase;		/* Base address of first SCC */
	int space;		/* Spacing between subsequent SCCs */
	int off[2];		/* Offset to A and B channel control regs */
	int doff;		/* Offset from control to data register */
	int ivec;		/* System interrupt vector number */
	long clk;		/* PCLK/RTxC frequency in Hz */
	int pclk;		/* flag to use PCLK (instead of RTxC) */
	int hwtype;		/* selection of special hardware types */
	int hwparam;		/* extra parameter for special hardware */

};
extern struct sccinfo sccinfo;

/* SCC channel control structure for AX.25 mode */
struct scca {
	unsigned int maxdefer;	/* Timer for CSMA defer time limit */

	unsigned int tstate;		/* Transmitter state */
#define IDLE		0	/* Transmitter off, no data pending */
#define DEFER		1	/* Receive Active - DEFER Transmit */
#define KEYUP		2	/* Permission to keyup the transmitter */
#define KEYWT		3	/* Transmitter switched on, waiting for CTS */
#define ACTIVE		4	/* Transmitter on, sending data */
#define FLUSH		5	/* CRC sent - attempt to start next frame */
#define TAIL		6	/* End of transmission, send tail */

	unsigned char txdelay;	/* Transmit Delay 10 ms/cnt */
	unsigned char persist;	/* Persistence (0-255) as a % */
	unsigned char slottime;	/* Delay to wait on persistence hit */
	unsigned char tailtime;	/* Delay after XMTR OFF */
	unsigned char fulldup;	/* Full Duplex mode 0=CSMA 1=DUP 2=ALWAYS KEYED */
	unsigned char waittime;	/* Waittime before any transmit attempt */
	unsigned char maxkeyup;	/* Maximum time to transmit (seconds) */
	unsigned char mintime;	/* Minimal offtime after MAXKEYUP timeout */
	unsigned char idletime;	/* Maximum idle time in ALWAYS KEYED mode (seconds) */

	/* statistical information of channel */
        long rxints;            /* Receiver interrupts */
        long txints;            /* Transmitter interrupts */
        long exints;            /* External/status interrupts */
        long spints;            /* Special receiver interrupts */
        
        long enqueued;          /* Packets actually forwarded */
        long rxframes;          /* Number of Frames Actally Received */
        long rxerrs;            /* CRC Errors */
        long txerrs;		/* KISS errors */
	unsigned int nospace;	/* "Out of buffers" */
	unsigned int rovers;	/* Receiver Overruns */
	unsigned int tovers;	/* Transmitter Overruns */

	int group;		/* group ID for AX.25 TX interlocking */
#define NORTXGROUP	0	/* not member of any group */
#define RXGROUP		0x100	/* if set, only tx when all channels clear */
#define TXGROUP		0x200	/* if set, don't transmit simultaneously */
	char tx_inhibit;	/* Transmit is not allowed when set */
	
	
#define	KISS_IDLE	0
#define KISS_DATA	1
#define KISS_ESCAPE	2

	char kiss_state;	/* state of the kiss interpreter */
	int bufsiz;
};

/* SCC channel structure. one is allocated for each attached SCC channel, */
/* so 2 of these are allocated for each fully utilized SCC chip */
struct sccchan {
	/* interrupt handlers for 4 different IP's */
	/* MUST BE first 4 elements of this structure, and MUST remain */
	/* in the sequence Transmit-Status-Receive-Special */
	void (*int_transmit)(register struct sccchan *scc);	/* Transmit Buffer Empty interrupt handler */
	void (*int_extstat)(register struct sccchan *scc);	/* External/Status Change interrupt handler */
	void (*int_receive)(register struct sccchan *scc);	/* Receive Character Avail. interrupt handler */
	void (*int_special)(register struct sccchan *scc);	/* Special Receive Condition interrupt handler */

	/* don't insert anything before "ctrl" (see assembly interrupt handler) */
	ioaddr ctrl;		/* I/O address of CONTROL register */
	ioaddr data;		/* I/O address of DATA register for this channel */

	unsigned char wreg[16]; /* Copy of last written value in WRx */
	unsigned char status;	/* Copy of R0 at last external interrupt */
	unsigned char txchar;	/* Char to transmit on next TX interrupt */

	int magic;		/* magic word */
	
        struct scca a;          /* control structure for AX.25 use */

	struct mbuf *sndq;	/* Packets awaiting transmission */
	struct mbuf *tbp;	/* Transmit mbuf being sent */


	char rxbuf[513];
	int  rxbufcnt;

	struct mbuf *sndq1;	/* Pointer to mbuf currently under construction */
	struct mbuf *sndq2;	/* Pointer to mbuf currently under construction */

	struct tty_struct *tty;
	unsigned char tty_opened;
	
	unsigned int timercount;/* 10ms timer for AX.25 use */

	long speed;		/* Line speed, bps */
	char extclock;		/* External clock source on RTxC/TRxC */
	char fulldup;		/* External divider for fulldup available */
	char nrz;		/* NRZ instead of NRZI */
	char enhanced;		/* Enhanced SCC support */
	char dum; 		/* filler (keep addr even for speed) */
};
extern struct sccchan *sccchan[];
#define NULLCHAN	(struct sccchan *)0

/* Z8530 SCC Register access macros */

#define rd(scc,reg)	RDSCC((scc)->ctrl,(reg))
#define wr(scc,reg,val) WRSCC((scc)->ctrl,(reg),((scc)->wreg[reg] = val))
#define or(scc,reg,val) WRSCC((scc)->ctrl,(reg),((scc)->wreg[reg] |= val))
#define cl(scc,reg,val) WRSCC((scc)->ctrl,(reg),((scc)->wreg[reg] &= ~(val)))



/* 8530 Serial Communications Controller Register definitions */
#define	FLAG	0x7e

/* Write Register 0 */
#define	R0	0		/* Register selects */
#define	R1	1
#define	R2	2
#define	R3	3
#define	R4	4
#define	R5	5
#define	R6	6
#define	R7	7
#define	R8	8
#define	R9	9
#define	R10	10
#define	R11	11
#define	R12	12
#define	R13	13
#define	R14	14
#define	R15	15

#define	NULLCODE	0	/* Null Code */
#define	POINT_HIGH	0x8	/* Select upper half of registers */
#define	RES_EXT_INT	0x10	/* Reset Ext. Status Interrupts */
#define	SEND_ABORT	0x18	/* HDLC Abort */
#define	RES_RxINT_FC	0x20	/* Reset RxINT on First Character */
#define	RES_Tx_P	0x28	/* Reset TxINT Pending */
#define	ERR_RES		0x30	/* Error Reset */
#define	RES_H_IUS	0x38	/* Reset highest IUS */

#define	RES_Rx_CRC	0x40	/* Reset Rx CRC Checker */
#define	RES_Tx_CRC	0x80	/* Reset Tx CRC Checker */
#define	RES_EOM_L	0xC0	/* Reset EOM latch */

/* Write Register 1 */

#define	EXT_INT_ENAB	0x1	/* Ext Int Enable */
#define	TxINT_ENAB	0x2	/* Tx Int Enable */
#define	PAR_SPEC	0x4	/* Parity is special condition */

#define	RxINT_DISAB	0	/* Rx Int Disable */
#define	RxINT_FCERR	0x8	/* Rx Int on First Character Only or Error */
#define	INT_ALL_Rx	0x10	/* Int on all Rx Characters or error */
#define	INT_ERR_Rx	0x18	/* Int on error only */

#define	WT_RDY_RT	0x20	/* Wait/Ready on R/T */
#define	WT_FN_RDYFN	0x40	/* Wait/FN/Ready FN */
#define	WT_RDY_ENAB	0x80	/* Wait/Ready Enable */

/* Write Register #2 (Interrupt Vector) */

/* Write Register 3 */

#define	RxENABLE	0x1	/* Rx Enable */
#define	SYNC_L_INH	0x2	/* Sync Character Load Inhibit */
#define	ADD_SM		0x4	/* Address Search Mode (SDLC) */
#define	RxCRC_ENAB	0x8	/* Rx CRC Enable */
#define	ENT_HM		0x10	/* Enter Hunt Mode */
#define	AUTO_ENAB	0x20	/* Auto Enables */
#define	Rx5		0x0	/* Rx 5 Bits/Character */
#define	Rx7		0x40	/* Rx 7 Bits/Character */
#define	Rx6		0x80	/* Rx 6 Bits/Character */
#define	Rx8		0xc0	/* Rx 8 Bits/Character */

/* Write Register 4 */

#define	PAR_ENA		0x1	/* Parity Enable */
#define	PAR_EVEN	0x2	/* Parity Even/Odd* */

#define	SYNC_ENAB	0	/* Sync Modes Enable */
#define	SB1		0x4	/* 1 stop bit/char */
#define	SB15		0x8	/* 1.5 stop bits/char */
#define	SB2		0xc	/* 2 stop bits/char */

#define	MONSYNC		0	/* 8 Bit Sync character */
#define	BISYNC		0x10	/* 16 bit sync character */
#define	SDLC		0x20	/* SDLC Mode (01111110 Sync Flag) */
#define	EXTSYNC		0x30	/* External Sync Mode */

#define	X1CLK		0x0	/* x1 clock mode */
#define	X16CLK		0x40	/* x16 clock mode */
#define	X32CLK		0x80	/* x32 clock mode */
#define	X64CLK		0xC0	/* x64 clock mode */

/* Write Register 5 */

#define	TxCRC_ENAB	0x1	/* Tx CRC Enable */
#define	RTS		0x2	/* RTS */
#define	SDLC_CRC	0x4	/* SDLC/CRC-16 */
#define	TxENAB		0x8	/* Tx Enable */
#define	SND_BRK		0x10	/* Send Break */
#define	Tx5		0x0	/* Tx 5 bits (or less)/character */
#define	Tx7		0x20	/* Tx 7 bits/character */
#define	Tx6		0x40	/* Tx 6 bits/character */
#define	Tx8		0x60	/* Tx 8 bits/character */
#define	DTR		0x80	/* DTR */

/* Write Register 6 (Sync bits 0-7/SDLC Address Field) */

/* Write Register 7 (Sync bits 8-15/SDLC 01111110) */

/* Write Register 8 (transmit buffer) */

/* Write Register 9 (Master interrupt control) */
#define	VIS	1	/* Vector Includes Status */
#define	NV	2	/* No Vector */
#define	DLC	4	/* Disable Lower Chain */
#define	MIE	8	/* Master Interrupt Enable */
#define	STATHI	0x10	/* Status high */
#define	NORESET	0	/* No reset on write to R9 */
#define	CHRB	0x40	/* Reset channel B */
#define	CHRA	0x80	/* Reset channel A */
#define	FHWRES	0xc0	/* Force hardware reset */

/* Write Register 10 (misc control bits) */
#define	BIT6	1	/* 6 bit/8bit sync */
#define	LOOPMODE 2	/* SDLC Loop mode */
#define	ABUNDER	4	/* Abort/flag on SDLC xmit underrun */
#define	MARKIDLE 8	/* Mark/flag on idle */
#define	GAOP	0x10	/* Go active on poll */
#define	NRZ	0	/* NRZ mode */
#define	NRZI	0x20	/* NRZI mode */
#define	FM1	0x40	/* FM1 (transition = 1) */
#define	FM0	0x60	/* FM0 (transition = 0) */
#define	CRCPS	0x80	/* CRC Preset I/O */

/* Write Register 11 (Clock Mode control) */
#define	TRxCXT	0	/* TRxC = Xtal output */
#define	TRxCTC	1	/* TRxC = Transmit clock */
#define	TRxCBR	2	/* TRxC = BR Generator Output */
#define	TRxCDP	3	/* TRxC = DPLL output */
#define	TRxCOI	4	/* TRxC O/I */
#define	TCRTxCP	0	/* Transmit clock = RTxC pin */
#define	TCTRxCP	8	/* Transmit clock = TRxC pin */
#define	TCBR	0x10	/* Transmit clock = BR Generator output */
#define	TCDPLL	0x18	/* Transmit clock = DPLL output */
#define	RCRTxCP	0	/* Receive clock = RTxC pin */
#define	RCTRxCP	0x20	/* Receive clock = TRxC pin */
#define	RCBR	0x40	/* Receive clock = BR Generator output */
#define	RCDPLL	0x60	/* Receive clock = DPLL output */
#define	RTxCX	0x80	/* RTxC Xtal/No Xtal */

/* Write Register 12 (lower byte of baud rate generator time constant) */

/* Write Register 13 (upper byte of baud rate generator time constant) */

/* Write Register 14 (Misc control bits) */
#define	BRENABL	1	/* Baud rate generator enable */
#define	BRSRC	2	/* Baud rate generator source */
#define	DTRREQ	4	/* DTR/Request function */
#define	AUTOECHO 8	/* Auto Echo */
#define	LOOPBAK	0x10	/* Local loopback */
#define	SEARCH	0x20	/* Enter search mode */
#define	RMC	0x40	/* Reset missing clock */
#define	DISDPLL	0x60	/* Disable DPLL */
#define	SSBR	0x80	/* Set DPLL source = BR generator */
#define	SSRTxC	0xa0	/* Set DPLL source = RTxC */
#define	SFMM	0xc0	/* Set FM mode */
#define	SNRZI	0xe0	/* Set NRZI mode */

/* Write Register 15 (external/status interrupt control) */
#define	ZCIE	2	/* Zero count IE */
#define	DCDIE	8	/* DCD IE */
#define	SYNCIE	0x10	/* Sync/hunt IE */
#define	CTSIE	0x20	/* CTS IE */
#define	TxUIE	0x40	/* Tx Underrun/EOM IE */
#define	BRKIE	0x80	/* Break/Abort IE */


/* Read Register 0 */
#define	Rx_CH_AV	0x1	/* Rx Character Available */
#define	ZCOUNT		0x2	/* Zero count */
#define	Tx_BUF_EMP	0x4	/* Tx Buffer empty */
#define	DCD		0x8	/* DCD */
#define	SYNC_HUNT	0x10	/* Sync/hunt */
#define	CTS		0x20	/* CTS */
#define	TxEOM		0x40	/* Tx underrun */
#define	BRK_ABRT	0x80	/* Break/Abort */

/* Read Register 1 */
#define	ALL_SNT		0x1	/* All sent */
/* Residue Data for 8 Rx bits/char programmed */
#define	RES3		0x8	/* 0/3 */
#define	RES4		0x4	/* 0/4 */
#define	RES5		0xc	/* 0/5 */
#define	RES6		0x2	/* 0/6 */
#define	RES7		0xa	/* 0/7 */
#define	RES8		0x6	/* 0/8 */
#define	RES18		0xe	/* 1/8 */
#define	RES28		0x0	/* 2/8 */
/* Special Rx Condition Interrupts */
#define	PAR_ERR		0x10	/* Parity error */
#define	Rx_OVR		0x20	/* Rx Overrun Error */
#define	CRC_ERR		0x40	/* CRC/Framing Error */
#define	END_FR		0x80	/* End of Frame (SDLC) */

/* Read Register 2 (channel b only) - Interrupt vector */

/* Read Register 3 (interrupt pending register) ch a only */
#define	CHBEXT	0x1		/* Channel B Ext/Stat IP */
#define	CHBTxIP	0x2		/* Channel B Tx IP */
#define	CHBRxIP	0x4		/* Channel B Rx IP */
#define	CHAEXT	0x8		/* Channel A Ext/Stat IP */
#define	CHATxIP	0x10		/* Channel A Tx IP */
#define	CHARxIP	0x20		/* Channel A Rx IP */

/* Read Register 8 (receive data register) */

/* Read Register 10  (misc status bits) */
#define	ONLOOP	2		/* On loop */
#define	LOOPSEND 0x10		/* Loop sending */
#define	CLK2MIS	0x40		/* Two clocks missing */
#define	CLK1MIS	0x80		/* One clock missing */

/* Read Register 12 (lower byte of baud rate generator constant) */

/* Read Register 13 (upper byte of baud rate generator constant) */

/* Read Register 15 (value of WR 15) */


/* 8536 register definitions */

#define CIO_MICR	0x00	/* Master interrupt control register */
#define CIO_MCCR	0x01	/* Master configuration control register */
#define CIO_CTMS1	0x1c	/* Counter/timer mode specification #1 */
#define CIO_CTMS2	0x1d	/* Counter/timer mode specification #2 */
#define CIO_CTMS3	0x1e	/* Counter/timer mode specification #3 */
#define CIO_IVR 	0x04	/* Interrupt vector register */

#define CIO_CSR1	0x0a	/* Command and status register CTC #1 */
#define CIO_CSR2	0x0b	/* Command and status register CTC #2 */
#define CIO_CSR3	0x0c	/* Command and status register CTC #3 */

#define CIO_CT1MSB	0x16	/* CTC #1 Timer constant - MSB */
#define CIO_CT1LSB	0x17	/* CTC #1 Timer constant - LSB */
#define CIO_CT2MSB	0x18	/* CTC #2 Timer constant - MSB */
#define CIO_CT2LSB	0x19	/* CTC #2 Timer constant - LSB */
#define CIO_CT3MSB	0x1a	/* CTC #3 Timer constant - MSB */
#define CIO_CT3LSB	0x1b	/* CTC #3 Timer constant - LSB */
#define CIO_PDCA	0x23	/* Port A data direction control */
#define CIO_PDCB	0x2b	/* Port B data direction control */

#define CIO_GCB 	0x04	/* CTC Gate command bit */
#define CIO_TCB 	0x02	/* CTC Trigger command bit */
#define CIO_IE		0xc0	/* CTC Interrupt enable (set) */
#define CIO_CIP 	0x20	/* CTC Clear interrupt pending */
#define CIO_IP		0x20	/* CTC Interrupt pending */


/* 8580/85180/85280 Enhanced SCC register definitions */

/* Write Register 7' (SDLC/HDLC Programmable Enhancements) */
#define AUTOTXF	0x01		/* Auto Tx Flag */
#define AUTOEOM 0x02		/* Auto EOM Latch Reset */
#define AUTORTS	0x04		/* Auto RTS */
#define TXDNRZI 0x08		/* TxD Pulled High in SDLC NRZI mode */
#define FASTDTR 0x10		/* Fast DTR/REQ Mode */
#define CRCCBCR	0x20		/* CRC Check Bytes Completely Received */
#define EXTRDEN	0x40		/* Extended Read Enabled */

/* Write Register 15 (external/status interrupt control) */
#define SHDLCE	1		/* SDLC/HDLC Enhancements Enable */
#define FIFOE	4		/* FIFO Enable */

/* Read Register 6 (frame status FIFO) */
#define BCLSB	0xff		/* LSB of 14 bits count */

/* Read Register 7 (frame status FIFO) */
#define BCMSB	0x3f		/* MSB of 14 bits count */
#define FDA	0x40		/* FIFO Data Available Status */
#define FOY	0x80		/* FIFO Overflow Status */


/* Device parameter control */

#define	PARAM_TXDELAY	1
#define	PARAM_PERSIST	2
#define	PARAM_SLOTTIME	3
#define	PARAM_TXTAIL	4
#define	PARAM_FULLDUP	5
#define PARAM_HW        6
#define PARAM_DTR       8
#define PARAM_SPEED     10
#define PARAM_GROUP     12
#define PARAM_IDLE      13
#define PARAM_MIN       14
#define	PARAM_MAXKEY	15
#define PARAM_WAIT      16
#define PARAM_TX        18


/* mbuf */

/* Basic message buffer structure */
struct mbuf {
	struct mbuf *next;	/* Links mbufs belonging to single packets */
	struct mbuf *anext;	/* Links packets on queues */
	int size;		/* Size of associated data buffer */
	int refcnt;		/* Reference count */
	struct mbuf *dup;	/* Pointer to duplicated mbuf */
/*	char *data; */		/* Active working pointers */
	char data[1024];	/* Active working pointers */
	int cnt;
	int in_use;		/* is this buffer in use ? */
};
#define	NULLBUF	(struct mbuf *)0
#define	NULLBUFP (struct mbuf **)0


/* global functions */

extern long scc_init(long kmem_start);


#endif	/* _SCC_H */
